/*
 * Copyright 1994 Phil Karn <karn@qualcomm.com>
 * Copyright 1996-1998 William A. Simpson <simpson@greendragon.com>
 * Copyright 2000 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Test probable "safe" primes,
 * suitable for use as Diffie-Hellman moduli;
 * that is, where q = (p-1)/2 is also prime.
 *
 * This is the second of two steps.  This step is processor intensive.
 *
 * 1996 May     William Allen Simpson
 *              extracted from earlier code by Phil Karn, April 1994.
 *              read large prime candidates list (q),
 *              and check prime probability of (p).
 * 1998 May     William Allen Simpson
 *              parameterized.
 *              optionally limit to a single generator.
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ssl/bn.h>

//#define FINALCHECK      1

/* need line long enough for largest prime and headers */
#define LINESIZE 32768

#define TRIAL_MINIMUM    (4)

#define TEST_MILLER_RABIN       (0x04)

typedef unsigned long int   uint32;     /* 32-bit unsigned integer */

int
main( int argc, char *argv[] )
{
        BIGNUM *q, *p, *a;
	BN_CTX *ctx;
        struct tm *gtm;
        FILE *finalfile = NULL;
        FILE *largefile = NULL;
        char *cp;
        char *lp;
        uint32 count_candidates = 0;
        uint32 count_safe = 0;
        uint32 generator_known;
        uint32 generator_wanted = 0;
        uint32 in_tests;
        uint32 in_tries;
        uint32 in_type;
        uint32 in_size;
        uint32 trials;
        time_t time_start;
        time_t time_stop;

        if (argc < 4) {
                printf("Missing argument: "
                       "<infile> <trials> <outfile> [<generator>]\n");
                exit(1);
        }
        largefile = fopen(argv[1],"r");
        if (largefile == NULL) {
                printf("Unable to read %s\n", argv[1] );
                exit(1);
        }
        if ((trials = strtoul(argv[2],NULL,10)) < TRIAL_MINIMUM)
                trials = TRIAL_MINIMUM;

        finalfile = fopen(argv[3],"a+");
        if ( finalfile == NULL ) {
                printf("Unable to write %s\n", argv[3] );
                exit(1);
        }
        if ( argc > 4 )
                generator_wanted = strtoul(argv[4],NULL,16);

        time(&time_start);

        p = BN_new();
        q = BN_new();
	ctx = BN_CTX_new();

        printf("%.24s Final %lu Miller-Rabin trials (%lx generator)\n",
                ctime(&time_start),
                trials,
                generator_wanted );

        lp = (char*)malloc(LINESIZE+1);

        while (fgets(lp,LINESIZE,largefile) != NULL) {
                int ll = strlen(lp);

                if ( ll < 14  ||  *lp == '!'  ||  *lp == '#' )
                        /* print("comment or blank\n"); */
                        continue;

                /* time */
                cp = &lp[14];   /* (skip) */

                /* type */
                in_type = strtoul( cp, &cp, 10 );

                // tests
                in_tests = strtoul( cp, &cp, 10 );
                if ( in_tests & 1 ) {
                        /* print("known composite\n"); */
                        continue;
                }

                /* tries */
                in_tries = strtoul( cp, &cp, 10 );

                /* size (bits) */
                in_size = strtoul( cp, &cp, 10 );
                if ( in_size < 63 ) {
                        /* print("short bit size\n"); */
                        continue;
                }

                /* generator (hex) */
                generator_known = strtoul( cp, &cp, 16 );

		/* Skip white space */
		cp += strspn(cp, " ");

                /* modulus (hex) */
                switch (in_type) {
                case 4:
                        /* print("Sophie-Germaine\n"); */
			a = q;
                        BN_hex2bn(&a, cp);

                        /* p = 2*q + 1 */
                        BN_lshift(p, q, 1);
                        BN_add_word(p, 1);

                        in_size += 1;
                        generator_known = 0;
                        break;
                default:
			a = p;
                        BN_hex2bn(&a, cp);

                        /* q = (p-1) / 2 */
                        BN_rshift(q, p, 1);
                        break;
                };

                if ( in_tests < TEST_MILLER_RABIN )
                        in_tries = trials;
                else
                        in_tries += trials;

                count_candidates++;

                /* guess unknown generator */
                if (generator_known == 0) {
                        if (BN_mod_word(p, 24) == 11 )
                                generator_known = 2;
                        else if (BN_mod_word(p, 12) == 5 )
                                generator_known = 3;
                        else {
                                u_int32_t r = BN_mod_word(p, 10);
                                if ( r == 3 || r == 7 ) {
                                        generator_known = 5;
                                }
                        }
                }

                /* skip tests when desired generator doesn't match */
                if (generator_wanted > 0 &&
		    generator_wanted != generator_known ) {
#ifdef  FINALCHECK
                        printf("generator %ld != %ld\n",
                                generator_known,
                                generator_wanted );
#endif
                        continue;
                }

                /* The (1/4)^N performance bound on Miller-Rabin is
                 * extremely pessimistic, so don't spend a lot of time
                 * really verifying that q is prime until after we know
                 * that p is also prime. A single pass will weed out the
                 * vast majority of composite q's.
                 */
		fprintf(stdout, "\n");

                if(BN_is_prime(q, 1, NULL, ctx, NULL) <= 0) {
#ifdef  FINALCHECK
                        printf("q failed first possible prime test\n");
#endif
                        continue;
                }

                /* q is possibly prime, so go ahead and really make
                 * sure that p is prime. If it is, then we can go back and
                 * do the same for q. If p is composite, chances are that
                 * will show up on the first Rabin-Miller iteration so
                 * it doesn't hurt to specify a high iteration count.
                 */
                if(!BN_is_prime(p, trials, NULL, ctx, NULL))
                {
#ifdef  FINALCHECK
                        printf("p is not prime\n");
#endif
                        continue;
                }
#ifdef  FINALCHECK
                printf("p is almost certainly prime\n");
#endif

                /* recheck q more rigorously */
                if(!BN_is_prime(q, trials - 1, NULL, ctx, NULL))
                {
#ifdef  FINALCHECK
                        printf("q is not prime\n");
#endif
                        continue;
                }
#ifdef  FINALCHECK
                printf("q is almost certainly prime\n");
#endif
                time(&time_stop);
                gtm = gmtime(&time_stop);
                fprintf(finalfile,"%04d%02d%02d%02d%02d%02d ",
                        gtm->tm_year + 1900,
                        gtm->tm_mon + 1,
                        gtm->tm_mday,
                        gtm->tm_hour,
                        gtm->tm_min,
                        gtm->tm_sec );
                fprintf(finalfile,"2 ");        /* Safe */
                fprintf(finalfile,"%lu ",in_tests | TEST_MILLER_RABIN);
                fprintf(finalfile,"%lu ",in_tries);
                fprintf(finalfile,"%lu ",in_size);
                fprintf(finalfile,"%lx ",generator_known);

                BN_print_fp(finalfile, p);
                fprintf(finalfile,"\n");
                count_safe++;
        }
        time(&time_stop);
        free(lp);
        fclose(finalfile);
        fclose(largefile);

	BN_free(p);
	BN_free(q);
	BN_CTX_free(ctx);

        printf("%.24s Found %lu safe primes of %lu candidates in %lu seconds\n",
                ctime(&time_stop),
                count_safe,
                count_candidates,
                (long)(time_stop - time_start) );

	exit(0);
}
