/*
 * Copyright 1994 Phil Karn <karn@qualcomm.com>
 * Copyright 1996-1998 William A. Simpson <simpson@greendragon.com>
 * Copyright 2000 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Sieve candidates for "safe" primes,
 * suitable for use as Diffie-Hellman moduli;
 * that is, where q = (p-1)/2 is also prime.
 *
 * This is the first of two steps.  This step is memory intensive.
 *
 * 1996 May     William Allen Simpson
 *              extracted from earlier code by Phil Karn, April 1994.
 *              save large primes list for later processing.
 * 1998 May     William Allen Simpson
 *              parameterized.
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ssl/bn.h>

// #define SMALL_CHECK   1
// #define LARGE_CHECK   1


/* Chosen for commonly available machines with at least 12 MB of
 * available main memory.  Using virtual memory can cause thrashing.
 * Therefore, this should be the largest number that is supported
 * without a large amount of disk activity -- that would increase the
 * run time from minutes to days or weeks!
 *** Do not increase this number beyond the unsigned integer bit size.
 *** Due to a multiple of 4, it must be LESS than 128 (yielding 2**30 bits).
 */
#define LARGE_MEMORY    (12UL)          /* megabytes */

/* Constant: When used with 32-bit integers, the largest sieve prime
 * has to be less than 2**32.
 */
#define SMALL_MAXIMUM    (0xffffffffUL)

/* Constant: can sieve all primes less than 2**32, as 65537**2 > 2**32-1.
 */
#define TINY_NUMBER     (1UL<<16)

/* Ensure enough bit space for testing 2*q.
 */
#define TEST_DOUBLER    (3)     /* 2**n, n < 5 */
#define TEST_LIMIT      (LARGE_MEMORY << ((18 - 6) - TEST_DOUBLER))
#define TEST_MAXIMUM    (TEST_LIMIT < (1L<<16) ? TEST_LIMIT : (1L<<16))
#define TEST_MINIMUM    (1L << 6)

/* Bit operations on 32-bit words
 */
#define BIT_CLEAR(a,n)  ((a)[(n)>>5] &= ~(1L << ((n) & 31)))
#define BIT_SET(a,n)    ((a)[(n)>>5] |= (1L << ((n) & 31)))
#define BIT_TEST(a,n)   ((a)[(n)>>5] & (1L << ((n) & 31)))

typedef unsigned long int   uint32;     /* 32-bit unsigned integer */

uint32 *LargeSieve;
uint32 largewords;
uint32 largetries;
uint32 largenumbers;
uint32 largebits;
BIGNUM *largebase;

uint32 *SmallSieve;
uint32 smallbits;
uint32 smallbase;

uint32 *TinySieve;
uint32 tinybits;


/* Sieve out p's and q's with small factors
 */
void
sieve_large( uint32 s )
{
        uint32 r;
        uint32 u;

#ifdef  SMALL_CHECK
        printf("%lu\n",s);
#endif
        largetries++;

        /* r = largebase mod s */
        r = BN_mod_word(largebase, s);
        if(r == 0)
        {
                /* s divides into largebase exactly */
                u = 0;
        }
        else
        {
                /* largebase+u is first entry divisible by s */
                u = s - r;
        }

        if ( u < largebits * 2 )
        {
                /* The sieve omits p's and q's divisible by 2,
                 * so ensure that largebase+u is odd.
                 * Then, step through the sieve in increments of 2*s
                 */
                if (u & 0x1)
                {
                        /* Make largebase+u odd, and u even */
                        u += s;
                }
                /* Mark all multiples of 2*s */
                for ( u /= 2; u < largebits; u += s )
                {
                        BIT_SET(LargeSieve,u);
                };
        }

        /* r = p mod s */
        r = (2*r+1) % s;
        if(r == 0)
        {
                /* s divides p exactly */
                u = 0;
        }
        else
        {
                /* p+u is first entry divisible by s */
                u = s - r;
        }

        if ( u < largebits * 4 )
        {
                /* The sieve omits p's divisible by 4,
                 * so ensure that largebase+u is not.
                 * Then, step through the sieve in increments of 4*s
                 */
                while (u & 0x3)
                {
                        if ( SMALL_MAXIMUM - u < s )
                        {
                                return;
                        }
                        u += s;
                };
                /* Mark all multiples of 4*s */
                for ( u /= 4; u < largebits; u += s )
                {
                        BIT_SET(LargeSieve,u);
                };
        }
}

int
main(int argc, char *argv[])
{
        BIGNUM *q;
        struct tm *gtm;
        FILE *largefile = NULL;
        uint32 j;
        uint32 power;
        uint32 r;
        uint32 s;
        uint32 smallwords = TINY_NUMBER >> 6;
        uint32 t;
        time_t time_start;
        time_t time_stop;
        uint32 tinywords = TINY_NUMBER >> 6;
        int i;

        if ( argc < 3 )
        {
                printf("Missing argument: "
                       "<outfile> <bits> [<start>]\n");
                exit(1);
        }

        /* Set power to the length in bits of the prime to be generated.
         * This is generally 1 less than the desired safe prime moduli p.
         */
        power = strtoul(argv[2],NULL,10);
        if ( power < TEST_MINIMUM )
        {
                printf("Too few bits: %lu < %lu\n",
                        power,
                        TEST_MINIMUM );
                exit(1);
        }
        else if ( power > TEST_MAXIMUM )
        {
                printf("Too many bits: %lu > %lu\n",
                        power,
                        TEST_MAXIMUM );
                exit(1);
        }

        largefile = fopen(argv[1],"a+");
        if ( largefile == NULL )
        {
                printf("Unable to write %s\n", argv[1] );
                exit(1);
        }

        if ( (TinySieve = (uint32*)calloc(tinywords,sizeof(uint32))) == NULL )
        {
                printf("Insufficient memory for tiny sieve: need %lu bytes\n",
                        tinywords << 2 );
                exit(1);
        }
        tinybits = tinywords << 5;              /* 32-bit words */

        if ( (SmallSieve = (uint32*)calloc(smallwords,sizeof(uint32))) == NULL )
        {
                printf("Insufficient memory for small sieve: need %lu bytes\n",
                        smallwords << 2 );
                free(TinySieve);
                exit(1);
        }
        smallbits = smallwords << 5;            /* 32-bit words */

        /* The density of ordinary primes is on the order of 1/bits,
         * so the density of safe primes should be about (1/bits)**2.
         * Set test range to something well above bits**2 to be reasonably
         * sure (but not guaranteed) of catching at least one safe prime.
         */
        largewords = ((power*power) >> (5-TEST_DOUBLER));
        if ( largewords > (LARGE_MEMORY << 18) )
        {
                largewords = (LARGE_MEMORY << 18);
        }

        /* dynamically determine available memory */
        while ( (LargeSieve = (uint32*)calloc(largewords,sizeof(uint32))) == NULL )
        {
                largewords -= (1L << 16);       /* 1/4 MB chunks */
        }
        largebits = largewords << 5;            /* 32-bit words */
        largenumbers = largebits << 1;          /* even numbers excluded */

        /* validation check: count the number of primes tried */
        largetries = 0;

        q = BN_new();
        largebase = BN_new();

        /* Generate random starting point for subprime search,
         * or use specified parameter.
         */
        if (argc < 4) {
                BN_rand(largebase, power, 1, 1);
        } else {
		BIGNUM *a;
		a = largebase;
                BN_hex2bn(&a, argv[3]);
        }

        /* ensure odd */
        if(!BN_is_odd(largebase))
                BN_set_bit(largebase, 0);

        time(&time_start);

        printf("%.24s Sieve large primes up to %lu from %lu-bit start point:\n",
                ctime(&time_start),
                largenumbers,
                power );
        BN_print_fp(stdout, largebase);
        printf("\n");

        /* TinySieve */
        for ( i = 0; i < tinybits; i++ ) {
                if(BIT_TEST(TinySieve,i))
                {
                        /* 2*i+3 is composite */
                        continue;
                }

                /* The next tiny prime */
                t = 2*i + 3;

                /* Mark all multiples of t */
                for( j = i+t; j < tinybits; j += t )
                {
                        BIT_SET(TinySieve,j);
                };
                sieve_large(t);
        };

        /* start the small block search at the next possible prime.
         * to avoid fencepost errors, the last pass is skipped.
         */
        for ( smallbase = TINY_NUMBER + 3;
              smallbase < (SMALL_MAXIMUM - TINY_NUMBER);
              smallbase += TINY_NUMBER )
        {
                for ( i = 0; i < tinybits; i++ )
                {
                        if(BIT_TEST(TinySieve,i))
                        {
                                /* 2*i+3 is composite */
                                continue;
                        }

                        /* The next tiny prime */
                        t = 2*i + 3;

                        r = smallbase % t;
                        if (r == 0)
                        {
                                /* t divides into smallbase exactly */
                                s = 0;
                        }
                        else
                        {
                                /* smallbase+s is first entry divisible by t */
                                s = t - r;
                        }

                        /* The sieve omits even numbers,
                         * so ensure that smallbase+s is odd.
                         * Then, step through the sieve in increments of 2*t
                         */
                        if (s & 1)
                        {
                                /* Make smallbase+s odd, and s even */
                                s += t;
                        }
                        /* Mark all multiples of 2*t */
                        for ( s /= 2; s < smallbits; s += t )
                        {
                                BIT_SET(SmallSieve,s);
                        };
                };

                /* SmallSieve */
                for ( i = 0; i < smallbits; i++ )
                {
                        if(BIT_TEST(SmallSieve,i))
                        {
                                /* 2*i+smallbase is composite */
                                continue;
                        }

                        /* The next small prime */
                        sieve_large( (2 * i) + smallbase );
                };
                memset( SmallSieve, 0, smallwords << 2 );
        };

        time(&time_stop);
        printf("%.24s Sieved with %lu small primes in %lu seconds\n",
                ctime(&time_stop),
                largetries,
                (long)(time_stop - time_start) );

        for(j=r=0;j<largebits;j++)
        {
                if(BIT_TEST(LargeSieve,j))
                {
                        /* Definitely composite, skip */
                        continue;
                }
#ifdef  LARGE_CHECK
                printf("test q = largebase+%lu\n",2*j);
#endif
		BN_set_word(q, 2*j);
                BN_add(q, q, largebase);

                time(&time_stop);
                gtm = gmtime(&time_stop);
                fprintf(largefile,"%04d%02d%02d%02d%02d%02d ",
                        gtm->tm_year + 1900,
                        gtm->tm_mon + 1,
                        gtm->tm_mday,
                        gtm->tm_hour,
                        gtm->tm_min,
                        gtm->tm_sec );
                fprintf(largefile,"4 ");        /* Sophie-Germaine */
                fprintf(largefile,"2 ");        /* Sieve */
                fprintf(largefile,"%lu ",largetries); /* iterations */
                fprintf(largefile,"%lu ",power); /* bits */
                fprintf(largefile,"0 ");        /* generator unknown */

                BN_print_fp(largefile, q);
                fprintf(largefile,"\n");
                r++;    /* count q */
        }
        time(&time_stop);
        free(LargeSieve);
        free(SmallSieve);
        free(TinySieve);
        fclose(largefile);

        printf("%.24s Found %lu candidates\n",
                ctime(&time_stop),
                r );

	exit(0);
}
